/*==============================================================================
 PART 1A: Deposit Volume & Interest Variable Construction
==============================================================================
Purpose:
Constructs key deposit-related variables from Call Report data, including:
- Standardized deposit categories (domestic, foreign, zero-maturity)
- Insured vs. uninsured deposit calculations
- Granular deposit and loan categorization (per maturity & holder type)
- Interest rate variables and ratios

Input:  Call Report data loaded in memory from master file
Output: Enhanced dataset with deposit and interest variables

Last updated: Aug 9, 2025
==============================================================================*/

display "--- Starting Part 1A: Deposit Volume & Interest Variable Construction ---"

//==============================================================================
// Section 1: Deposit Volume & Interest Variable Construction
//==============================================================================

//------------------------------------------------------------------------------
// 1.1. Standardize Core Deposit Variables
//------------------------------------------------------------------------------
// Assume foreign deposits are zero if missing
replace foreigndep = 0 if missing(foreigndep)

// Identify zero-maturity deposits (transaction + savings deposits)
gen zmdep = transdep+savdep

//------------------------------------------------------------------------------
// 1.2. Uninsured/Insured Deposit Values
//------------------------------------------------------------------------------
// Uninsured deposits are primarily based on the self-reported value (uninsureddep, RC-O item, RCON5597).
// When self-reported uninsured deposits are missing, we impute insured deposits (and thus uninsured)
// based on the amount in accounts below $250k and the number of accounts above $250k,
// assuming $250k is insured per account above the threshold.

// Use self-reported uninsured deposits ('uninsureddep', RCON5597).
// If missing, impute uninsured deposits as total domestic deposits minus the calculated insured amount.
gen uninsureddep = uninsureddep2
replace uninsureddep = domdep - insureddep if uninsureddep == .

// Update insured deposits based on the final uninsureddep value.
replace insureddep = domdep - uninsureddep
replace insureddep = 0 if insureddep < 0 // Set negative imputed insured values to zero

//------------------------------------------------------------------------------
// 1.3. Interest Rate Variables
//------------------------------------------------------------------------------
// Calculate annualized deposit expense rates by deposit type
// Note: Multiply by 4 to annualize quarterly rates

// Interest expenses on "zero maturity" deposits (transaction + savings)
gen zmdepexprate = (intexpsavdep + intexptransdep) / (savdep + transdep) * 4
// Interest expenses on all domestic deposits
gen depexprate = intexpdomdep / domdep * 4

// Calculate annualized interest expense and income rates relative to quarterly average assets
gen intexprate = intexp / qavgassets * 4
gen intincrate = intincassets / qavgassets * 4

// Non-interest income and expense variables (used in cost estimations)
gen nonintexp_assets=nonintexp*400/qavgassets // Annualized percentage of average assets
gen nonintinc_assets=nonintinc*400/qavgassets // Annualized percentage of average assets
gen net_nonintexp_assets=nonintexp_assets-nonintinc_assets

display "--- Part 1A completed ---"
display "Deposit and interest variables constructed"
